#Multilevel modeling in R!!!

#First we're going to begin by installing or activating the packages we are going to need
#install.package("lme4")
library(lme4)
install.packages("multilevel")
library(multilevel)
#install.packages("ggplot2") 
install.packages('lmerTest')
library(lmerTest)
library(ggplot2)
install.packages('car')
library(car)
#Some of these packages are useful for multilevel modeling itself, others are useful for auxillary information
#Let's start with a review of OLS regression, many of the concepts will show up in the multilevel context
#We are going to begin by activating one of the datasets we will be working with today
data(bh1996)
#This is the first dataset we will be working with
attach(bh1996)
View(bh1996)
#Here we're specifying a simple one predictor regression model
ols <- lm(LEAD ~ HRS)
#Let's take a look at some summary statistics
summary(ols)
#Let's think a little bit about what this output means
anova(ols)
#Let's take a look at everything we can learn from this model
attributes(ols)
#Let's check some assumptions
residuals <- ols$residuals
plot(residuals)
plot(bh1996$LEAD, residuals)
ols$fitted.values
residualPlots(ols)
qqPlot(ols)
#Many of these concepts will also be used in multievle modeling
#Let's move onto to specifying an unconditional multilevel model

unconditional <- lmer(HRS~ 1 + (1|GRP), data=bh1996)
summary (unconditional)
#We can learn quite a bit from this model
#First is information related to the variance terms
hrs.mod<-aov(HRS~as.factor(GRP),data=bh1996)
#ICC1 calculates the actual ICC
ICC1(hrs.mod)
#We can calculate it by hand just to make sure
icc <- .67/(.67+4.48)
icc
#ICC2 calculates the relaibility of the intercept
ICC2(hrs.mod)
#We can also bootstrap the ICC
bootoutput <-boot.icc (bh1996$HRS, grpid=bh1996$GRP, nboot=100)
#And calculate the CI around that value
quantile(bootoutput, c(.025, .975))
#We can find multiple ICCs for multiple variables
mult.icc(bh1996[,c("HRS", "LEAD")], grpid = bh1996$GRP)
#If we can to know significance of our random effects
rand(unconditional)
#We can find all the modifications to our lmer model here
?lmer
#Let's add some predictors to our model now
onepredictorfixed <- lmer(HRS~LEAD + (1|GRP), data=bh1996)
summary(onepredictorfixed)
#Let's look at what the output is telling us
#The model we just specified treats the LEAD effects as fixed, let's specify it randomly
onepredictorrandom <- lmer(HRS~LEAD + (LEAD|GRP), data=bh1996)
summary(onepredictorrandom)
#Let's check out the signficance of the random effect
rand(onepredictorrandom)
#Now using the anova function
anova(onepredictorfixed, onepredictorrandom)
#Notice that we are using up two degrees of freedom. That's because a covariance among random effects is specified
#Let's specify uncorrelated random effects
onepredictoruncorr <- lmer(HRS~LEAD + (1|GRP) + (0+LEAD|GRP), data=bh1996)
summary(onepredictoruncorr)
#Now we can compare models
anova(onepredictoruncorr, onepredictorrandom)
#Let's take a closer look at the estimated effects
coef(onepredictorfixed)
coef(onepredictorrandom)
coef(onepredictoruncorr)
#Now we can take a closer look at the some of the information about the model
#Like the convergence criterion
REMLcrit(onepredictoruncorr)
#Or the family of effects
family(onepredictoruncorr)
#We can take a look at some of the boostrap confidence intervals for our model
#Remember, more simultations will mean more computer time
confint(onepredictoruncorr, method="boot", nsim=200)
#Now if we wanted to specify a random slope but not intercept
onepredictor <- lmer(HRS~LEAD + (0+ LEAD|GRP), data=bh1996)
summary(onepredictor)

#Often times we don't want just one predictor, but want to test the effect of multiple IVs
#Again here we are just specifying fixed slopes
twopredictor <- lmer(HRS~LEAD + COHES + (1|GRP), data=bh1996)
summary(twopredictor)
#Again let's take a quick look at the output
#Now let's specify the Cohesion slope as random
COrandom <- lmer(HRS~LEAD + COHES + (COHES|GRP), data=bh1996)
summary(COrandom)
#Notice the difference in degrees of freedom!
#the default in lmerTest is to use Satterthwaite approximations, so every predictor will have different df's
#Let's examine significance of the random effects
rand(COrandom)
#And the coefficient values
coef(COrandom)
#Now let's specify both slopes as random
tworandom <- lmer(HRS~LEAD + COHES + (LEAD + COHES|GRP), data=bh1996)
summary(tworandom)
rand(tworandom)
#According to this test, bothg random effects are non-significant
#Look at the df's now, they're adding up. Let's specify all uncorrelated random effects
tworandomuncorr <- lmer(HRS~LEAD + COHES + (0+COHES|GRP)+(0+LEAD|GRP) + (1|GRP), data=bh1996)
summary(tworandomuncorr)
rand(tworandomuncorr)
#We get a different picture regarding significance of random effects, but it looks like Cohesion is still not significant
final <- lmer(HRS~LEAD + COHES +(0+LEAD|GRP) + (1|GRP), data=bh1996)
summary(final)
rand(final)
#This appears to be the model that best balances parsimony and prediction

#Let's add a level two predictor now
#Now we're adding the same variable, just the group instead 
leveltwo <- lmer(HRS~G.LEAD + (1|GRP), data=bh1996)
summary(leveltwo)
confint(leveltwo, method="boot", nsim=200)
#Let's see what happens when we try to treat this as random...
leveltworand <- lmer(HRS~G.LEAD + (G.LEAD|GRP), data=bh1996)
summary(leveltworand)
#Look at the correlation of random effects now
anova(leveltwo, leveltworand)
#Adding that random effect does nothing for the model
#Let's now try adding one predictor at each level
bothlevels <- lmer (HRS~LEAD + G.LEAD + (1|GRP), data=bh1996)
summary(bothlevels)
#We can see that both individual and group leadership is associated with Hours
#This group mean mights be a bit redundant with every person's score
within <- lmer (HRS~W.LEAD + G.LEAD + (1|GRP), data=bh1996)
summary(within)
#Look at the correlation between within and group
#Now we should specifcy a random effect of within
withinrand <- lmer (HRS~W.LEAD + G.LEAD + (W.LEAD|GRP), data=bh1996)
summary(withinrand)
coef(withinrand)
#Looking at the significance of the random effect
anova(within, withinrand)

#Let's look now at a cross-level interaction
#This is a model where we are essentially using the group mean to predict individual effect
int <- lmer (HRS~W.LEAD * G.LEAD + (1|GRP), data=bh1996)
summary(int)
#So there isn't a significant interaction here, but we should specify the random effect anyway
randint <- lmer (HRS~W.LEAD * G.LEAD + (W.LEAD|GRP), data=bh1996)
summary(randint)
anova(int, randint)
#The random effects model still fits better

#Graphing
#Because of the nature of nested data, there aren't a ton of good graphing options
#Let's start with a simple scatterplot
plot(HRS~LEAD)
#This isn't terribly useful, let's add a best fit line
abline(lm(HRS~LEAD), col="Red")
#Both of these graphs ignore the nest component of our data. One option is to examine individual groups
#This line creates a separate data frame from Group 12
GRP.12 <- bh1996[bh1996$GRP==12, ]
#And we can see the scatterplot for group 12
plot(GRP.12$HRS~GRP.12$LEAD)
abline(lm(GRP.12$HRS~GRP.12$LEAD))
#We can make a "spaghetti plot", while essentially created a scatter plot for each group
ggplot(bh1996, aes(x=WBEING, y = HRS)) + geom_line(aes(group = GRP))
spaghetti <- ggplot(bh1996, aes(x=WBEING, y = HRS)) + geom_line(aes(group = GRP, color = as.factor(GRP)))
spaghetti
spaghetti  + theme(legend.position = 'none')
spaghetti + stat_smooth(method = 'lm') + theme(legend.position = 'none')
spaghetti + stat_smooth(method = 'lm') +theme_minimal() + theme(legend.position = 'none')

#Longitudinal Modeling
#To begin, we need to use a new dataset
data("tankdat")
attach(tankdat)
View(tankdat)
?tankdat
#Like before, we will begin by specifying an unconditional model
unconditional <- lmer(SCORE~1 + (1|ID), data=tankdat)
summary(unconditional)
#We should also be calculating the ICCs
icc <- 46.16/(46.16+58.59)
icc
#Or remember...
uncondlong <- aov(SCORE~ as.factor(ID), data=tankdat)
#Specify the ICC
ICC1(uncondlong)
#Specify the reliability of the intercept
ICC2(uncondlong)
#Now we can specify the growth model portion of these data
growth <- lmer(SCORE~TIME + (1|ID), data=tankdat)
summary(growth)
#WE can see a significant effect of time. We can specify the effect of time as random
growthrand <- lmer(SCORE~TIME + (TIME|ID), data=tankdat)
summary(growthrand)
#The correlation between random effects is especially worth noting here
#Let's check if the random effect is significant
anova(growth, growthrand)
#What about the covariance between random effects
growthuncorrelated <- lmer(SCORE~TIME + (0+TIME|ID) + (1|ID), data=tankdat)
summary(growthuncorrelated)
anova(growthuncorrelated, growthrand)
#Both effects are definitely significant
#We can also add time-invariant predictors
growthcon <- lmer(SCORE~CONSC  + (1|ID), data=tankdat)
summary(growthcon)
#So conscientiousness doesn't predict scores on this test, does it predict the slope of time?
growthint <- lmer(SCORE~TIME*CONSC  + (1|ID), data=tankdat)
summary(growthint)
#The nonsignificant interaction tells us the slope doesn't vary based on conscientiousness
#We should still try a random effects model
growthintrand <- lmer(SCORE~TIME*CONSC  + (TIME|ID), data=tankdat)
summary(growthintrand)
#Lastly, when dealing with interactions like this, it may be a good idea to center predictors
#The scale function can be used to center or z-score variables. The "scale=FALSE" command centers the variables w/o z-scoring
tankdat$TIME_c <- scale(tankdat$TIME, scale=FALSE)
tankdat$CONSC_c <- scale(tankdat$CONSC, scale=FALSE)
#Now let's run the same model with our centered variables
growthintcenter<- lmer(SCORE~TIME_c*CONSC  + (TIME_c|ID), data=tankdat)
summary(growthintcenter)
#We can see that some of the estimates changed, but the inference wasn't really affected

#Graphing
#Because of the nature of nested data, there aren't a ton of good graphing options
#Let's start with a simple scatterplot
plot(HRS~LEAD)
#This isn't terribly useful, let's add a best fit line
abline(lm(HRS~LEAD), col="Red")
#Both of these graphs ignore the nest component of our data. One option is to examine individual groups
#This line creates a separate data frame from Group 12
GRP.12 <- bh1996[bh1996$GRP==12, ]
#And we can see the scatterplot for group 12
plot(GRP.12$HRS~GRP.12$LEAD)
abline(lm(GRP.12$HRS~GRP.12$LEAD))
#We can make a "spaghetti plot", while essentially created a scatter plot for each group
ggplot(bh1996, aes(x=WBEING, y = HRS, group = GRP, color = GRP)) + geom_line()
spaghetti <- ggplot(bh1996, aes(x=WBEING, y = HRS)) + geom_line(aes(group = GRP, color = as.factor(GRP)))
spaghetti
#After creating our graphical object, we can make modifications to it
spaghetti + stat_smooth(method = 'lm') +theme_minimal() + theme(legend.position = 'none')
#Longitudinal example
ggplot(tankdat, aes(x = TIME, y = SCORE, color = as.factor(ID))) + geom_smooth(method="loess", fill=NA)
#This gives us a visualization of change over time, but looks pretty intense
ggplot(tankdat, aes(x = TIME, y = SCORE, color = as.factor(ID))) + geom_smooth(method="loess", fill=NA) + theme(legend.position = 'none')
#This method will instead fit a straight line for everyone, giving an impression of lienar growth
ggplot(tankdat, aes(x = TIME, y = SCORE, color = as.factor(ID))) + geom_smooth(method="lm", fill=NA) + theme(legend.position = 'none')
#Finally, we're changing the theme to one I like
ggplot(tankdat, aes(x = TIME, y = SCORE, group = as.factor(ID))) + geom_smooth(method="lm", fill=NA) + theme(legend.position = 'none') + theme_bw()

#The Logistic extension
#To demonstrate logistic multilevel models, we need yet another dataset
data("VerbAgg")
attach(VerbAgg)
View(VerbAgg)
#To run a logistic regression model we need specify the glm extension of lm command
logistic <- glm(r2~Anger, family="binomial", data=VerbAgg)
summary(logistic)
#To get an odds-ratio from our output
exp(.034972)
#now let's take a look at why dichotomous outcomes violate our assumptions
resid(logistic)
#We can't see a ton just from looking at our residuals. Let's plot now
plot(resid(logistic))
hist(resid(logistic))
#We can definitely see now that our residuals are not normally distributed
#the qqplots tell the same story
qqnorm(resid(logistic))
qqline(resid(logistic))
# we can also see something with our predicted values
fitted.values (logistic)
#Let's look at some of our other output options
attributes(logistic)
#We can see how well our baseline model fits
logistic$null.deviance
#We can see how our fitted model fits
logistic$deviance
#One really important aspect
logistic$converged
#How many iterations to convergence
logistic$iter

#The same basic principles of logistic regression apply to multilevel models as well, with a few modifications
logisticmlm <- glmer(r2~Anger + (1|id),family="binomial", data=VerbAgg)
summary(logisticmlm)
#The "family" function is what's really new here, of which we have severla options
?family
#As with any multilevel model, we can specify random slopes
logisticrandom <- glmer(r2~Anger + (Anger|id),family="binomial", data=VerbAgg)
summary(logisticrandom)
#We can see our model did not converge, we can try simplifying the model a bit
logisticrandom <- glmer(r2~Anger + (1|id) + (0+Anger|id),family="binomial", data=VerbAgg)
summary(logisticrandom)
#However we can still get some info
fitted.values(logisticmlm)
fixef(logisticmlm)
deviance(logisticmlm)
#In light of the model not converging, let's take a look at some of the options open to us
#to try and modify the model so that it ends up running no problem
#This gives us the different modifications we can make for the estimation
?glmerControl
#The default optimizer for lmer is bobyqa, for glmer it starts as bobyqa and switches to Nelder-Mead
#bobyqa is short for bound optimization by quadratic approximation, Nelder-Mead doesn't involve known derivatives like
#bobyqa, so there are times when this optimization technique works better.
logisticrandom <- glmer(r2~Anger + (1|id) + (0+Anger|id),family="binomial", data=VerbAgg)
Nelderlogistic <- glmer(r2~Anger + (1|id) + (0+Anger|id),family="binomial", data=VerbAgg, control = glmerControl(optimizer = 'Nelder_Mead'))
#We can see that using just the Nelder-Mead optimizer didn't help us. We can try just the bobyqa method
bobylogistic <- glmer(r2~Anger + (1|id) + (0+Anger|id),family="binomial", data=VerbAgg, control = glmerControl(optimizer = 'bobyqa'))
#Now we got rid of our error message, this model appears to have converged fine
bobylogistic
logisticrandom
#However looking at the parameter estimates and the model fit, it doesn't look like non-convergence really mattered
#One possibility is that the model is simply too complex, and we should roll back expectations for tolerance
logisticrandom <- glmer(r2~Anger + (1|id) + (0+Anger|id),family="binomial", data=VerbAgg, control = glmerControl(boundary.tol = 1e-3))
#One feature of these optimization techniques is that they calculate what are called the 'Jacobian' and 'Hessian'
#matrices. These are the second and third order partial derivates of the function. The calculation of these is often the 
#source of the problem. We cna turn that off.
logisticrandom <- glmer(r2~Anger + (1|id) + (0+Anger|id),family="binomial", data=VerbAgg, control = glmerControl(calc.derivs = FALSE))
summary(logisticrandom)
#So again things seem to work just fine, we got rid of the converge errors. However using optimization without the Jacobian
#or Hessian, our standard errors are less accurate (they're calculated for a reason!)
?glmer
#The last modification I want to talk about is the nAGQ argument. This argument specifies the
#number of Adaptive Gaussian Quadratures. This essentially specifies how precise we want the integration of the function to be.
#This can often solve our issue with the caveats that it only works with a relatively simple random effects structure
#AND increasing this number rapidly escalates our computing time.
logisticrandom <- glmer(r2~Anger + (1|id),family="binomial", data=VerbAgg, nAGQ = 10)
logisticrandom
